<?php

namespace App\Custom\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Annotation\Route;

class LegacyEntryPointController extends AbstractController
{
    /**
     * Handle legacy entry points for contact operations
     * 
     * INSERT API Example:
     * /public/index.php?id=3&entryPoint=CreateContact&username=zittubhai69&lastname=zittubhai69&phone=01933608
     * 
     * UPDATE API Example:
     * /public/index.php?entryPoint=UpdateContact&username=zittubhai69
     */
    #[Route('/public/index.php', name: 'legacy_entry_point', methods: ['GET', 'POST'])]
    public function handleEntryPoint(Request $request): JsonResponse
    {
        $entryPoint = $request->query->get('entryPoint');
        
        if (!$entryPoint) {
            return new JsonResponse([
                'error' => 'Missing entryPoint parameter',
                'available_entry_points' => ['CreateContact', 'UpdateContact']
            ], 400);
        }

        switch ($entryPoint) {
            case 'CreateContact':
                return $this->createContact($request);
            case 'UpdateContact':
                return $this->updateContact($request);
            default:
                return new JsonResponse([
                    'error' => 'Invalid entryPoint: ' . $entryPoint,
                    'available_entry_points' => ['CreateContact', 'UpdateContact']
                ], 400);
        }
    }

    /**
     * Create Contact API
     * Example: /public/index.php?id=3&entryPoint=CreateContact&username=zittubhai69&lastname=zittubhai69&phone=01933608
     */
    private function createContact(Request $request): JsonResponse
    {
        $username = $request->query->get('username');
        $lastname = $request->query->get('lastname');
        $phone = $request->query->get('phone');
        $id = $request->query->get('id');

        if (!$username || !$lastname) {
            return new JsonResponse([
                'error' => 'Missing required fields: username and lastname are required',
                'received_data' => [
                    'username' => $username,
                    'lastname' => $lastname,
                    'phone' => $phone,
                    'id' => $id
                ]
            ], 400);
        }

        try {
            // Load SuiteCRM legacy
            require_once __DIR__ . '/../../../public/legacy/include/entryPoint.php';

            // Create new Contact bean
            $contact = \BeanFactory::newBean('Contacts');
            $contact->first_name = $username;
            $contact->last_name = $lastname;
            $contact->phone_mobile = $phone;
            $contact->call_status_c = "Haven't Called"; // Set initial call status
            $contact->tag_c = "Hot"; // Set initial tag

            $contact->save();

            return new JsonResponse([
                'success' => true,
                'message' => 'Contact created successfully',
                'data' => [
                    'id' => $contact->id,
                    'username' => $username,
                    'lastname' => $lastname,
                    'phone' => $phone,
                    'call_status' => "Haven't Called",
                    'tag' => "Hot"
                ]
            ]);

        } catch (\Exception $e) {
            return new JsonResponse([
                'error' => 'Failed to create contact',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update Contact API
     * Example: /public/index.php?entryPoint=UpdateContact&username=zittubhai69
     */
    private function updateContact(Request $request): JsonResponse
    {
        $username = $request->query->get('username');

        if (!$username) {
            return new JsonResponse([
                'error' => 'Missing required field: username',
                'received_data' => [
                    'username' => $username
                ]
            ], 400);
        }

        try {
            // Load SuiteCRM legacy
            require_once __DIR__ . '/../../../public/legacy/include/entryPoint.php';

            $contactBean = \BeanFactory::newBean('Contacts');
            $contacts = $contactBean->get_full_list(
                '', "contacts.first_name = '{$username}'"
            );

            if (empty($contacts)) {
                return new JsonResponse([
                    'error' => 'Contact not found',
                    'searched_username' => $username
                ], 404);
            }

            $contact = $contacts[0];
            $contact->call_status_c = "Deposit Done"; // Update call status
            $contact->tag_c = "VIP"; // Update tag to VIP

            $contact->save();

            return new JsonResponse([
                'success' => true,
                'message' => 'Contact updated successfully',
                'data' => [
                    'id' => $contact->id,
                    'username' => $username,
                    'call_status' => "Deposit Done",
                    'tag' => "VIP"
                ]
            ]);

        } catch (\Exception $e) {
            return new JsonResponse([
                'error' => 'Failed to update contact',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
